# ---------------------------------------------------------------------------------------------------------------------
# coding: UTF-8
# author:Yoshitaka Yoshida(jj,micropower,HigashiHiroshima City,Museum of Yoyo major,2019)
# ---------------------------------------------------------------------------------------------------------------------

# import sys
import math
import argparse
import os
from chardet.universaldetector import UniversalDetector

PROGRAM_VERSION = "1.07"
PROGRAM_UPDATE_TIME = "May 6,2019"
PROGRAM_AUTHOR = "Yoyo"

# --------------------------------------- #
# ucode : ubiquitous code format constant #
# --------------------------------------- #
VER_WIDTH = 4
TLDC_WIDTH = 16
CC_WIDTH = 4
SLDC_PLUS_IC_WIDTH = 104
VER_START = 0
TLDC_START = VER_WIDTH
CC_START = VER_WIDTH + TLDC_WIDTH
SLDC_START = CC_START + CC_WIDTH
TLDC_META_CODE_SPACE = 0xe000
TLDC_RESERVED_SPACE = 0xefff
TLDC_COMMON_VOCABULARY = 0xfffd
TLDC_ETRON_ID = 0xffff
CC_CLASS_RESERVED_1000 = 0b1000
CC_CLASS_A = 0b1001
CC_CLASS_B = 0b1010
CC_CLASS_C = 0b1011
CC_CLASS_D = 0b1100
CC_CLASS_E = 0b1101
CC_CLASS_F = 0b1110
CC_CLASS_RESERVED_1111 = 0b1111
# ------------------------------------------------------------------------------------ #
# 国土地理院(GSI) 場所情報コード:uPlace(ucode for Place indentification)　フォーマット #
# ------------------------------------------------------------------------------------ #
UPLACE_CLASSIFICATION_START = 0
UPLACE_CLASSIFICATION_WIDTH = 2
UPLACE_LATITUDE_START = UPLACE_CLASSIFICATION_START + UPLACE_CLASSIFICATION_WIDTH
UPLACE_LATITUDE_WIDTH = 23
UPLACE_LATITUDE_SIGN_WIDTH = 1
UPLACE_LATITUDE_VALUE_START = UPLACE_LATITUDE_START + UPLACE_LATITUDE_SIGN_WIDTH
UPLACE_LATITUDE_VALUE_WIDTH = UPLACE_LATITUDE_WIDTH - UPLACE_LATITUDE_SIGN_WIDTH
UPLACE_LONGITUDE_START = UPLACE_LATITUDE_START + UPLACE_LATITUDE_WIDTH
UPLACE_LONGITUDE_WIDTH = 24
UPLACE_LONGITUDE_SIGN_WIDTH = 1
UPLACE_LONGITUDE_VALUE_START = UPLACE_LONGITUDE_START + UPLACE_LONGITUDE_SIGN_WIDTH
UPLACE_LONGITUDE_VALUE_WIDTH = UPLACE_LONGITUDE_WIDTH - UPLACE_LONGITUDE_SIGN_WIDTH
UPLACE_FLOOR_START = UPLACE_LONGITUDE_START + UPLACE_LONGITUDE_WIDTH
UPLACE_FLOOR_WIDTH = 8
UPLACE_BETWEEN_FLOOR_START = UPLACE_FLOOR_START + UPLACE_FLOOR_WIDTH
UPLACE_BETWEEN_FLOOR_WIDTH = 1
UPLACE_IDENTIFICATION_NO_START = UPLACE_BETWEEN_FLOOR_START + UPLACE_BETWEEN_FLOOR_WIDTH
UPLACE_IDENTIFICATION_NO_WIDTH = 6
# ---------------------------------- #
# ISO/IEC 11784 コード　フォーマット #
# ---------------------------------- #
ANIMAL_RFID_ANIMAL_START = 0
ANIMAL_RFID_ANIMAL_WIDTH = 1
ANIMAL_RFID_RESERVED_START = ANIMAL_RFID_ANIMAL_START + ANIMAL_RFID_ANIMAL_WIDTH
ANIMAL_RFID_RESERVED_WIDTH = 14
ANIMAL_RFID_DATA_BLOCK_START = ANIMAL_RFID_RESERVED_START + ANIMAL_RFID_RESERVED_WIDTH
ANIMAL_RFID_DATA_BLOCK_WIDTH = 1
ANIMAL_RFID_COUNTRY_START = ANIMAL_RFID_DATA_BLOCK_START + ANIMAL_RFID_DATA_BLOCK_WIDTH
ANIMAL_RFID_COUNTRY_WIDTH = 10
ANIMAL_RFID_IDENTIFICATION_CODE_START = ANIMAL_RFID_COUNTRY_START + ANIMAL_RFID_COUNTRY_WIDTH
ANIMAL_RFID_IDENTIFICATION_CODE_WIDTH = 38
# --------------------------------------------------------------------------------- #
# 会員向けucodeドメイン割当て　(2006年12月20日幹事会承認） See TRONWARE VOL.103,p86 #
# --------------------------------------------------------------------------------- #
MD_G = "general"
MD_P = "provider"
MEMBER_DOMAIN_YRP = "YRP Ubiquitous Networking Laboratory", MD_G, "0-0001-C-000000-00000001-************"
MEMBER_DOMAIN_UCTECH = "Ubiquitous Computing Technology Corporation", MD_P, "0-0001-C-000000-00000002-************"
MEMBER_DOMAIN_S = "Sakamura-Lab, The University of Tokyo", MD_G, "0-0001-C-000000-00000003-************"
MEMBER_DOMAIN_K = "Koshizuka-Lab, The University of Tokyo", MD_G, "0-0001-C-000000-00000004-************"
MEMBER_DOMAIN_UNIADEX = "UNIADEX, Ltd.", MD_P, "0-0001-C-000000-00000005-************"
MEMBER_DOMAIN_MTI = "MTI Co., Ltd.", MD_G, "0-0001-C-000000-00000006-************"
MEMBER_DOMAIN_JTA = "Japan Traceability Association", MD_P, "0-0001-B-00-00000001-****************"
MEMBER_DOMAIN_FBSEAJ = "Fishing Boat And System Engineering Association of Japan", MD_P, \
                  "0-0001-B-00-00000002-****************"

def exchange_pattern(entry_string):
    """
    ucodeのパターン比較文字列を与えて、成否と文字ごとの１６進数配列と比較配列（真は比較、偽は任意）を返す。
    :param entry_string: ucode16進数文字列(-は無視、*は任意を示す）
    :return:
    """
    result = False
    pat = []
    flag = []
    pat_str = entry_string.lower().replace("-", "")
    if len(pat_str)*4 != 128:
        return result, pat, flag
    hex_digits = set("0123456789abcdef")
    result = True
    for ch in pat_str:
        if ch == "*":
            flag.append(False)
        else:
            flag.append(True)
            if ch in hex_digits:
                pat.append(int(ch, 16))
            else:
                pat.append(0)
                result = False
    return result, pat, flag


def display_bit_with_unit(value):
    """
    単位付きのビット数表現を返す。
    :param value: value(bit数）
    :return: value(0,1)--->0bit,1bit, value(>1)bits
    """
    if abs(value) <= 1:
        return str(value) + "bit"
    else:
        return str(value) + "bits"

def del_hex_prefix(data_str):
    """
    16進数表記の先頭語0xの表記を削除する。
    :param data_str: string
    :return: string
    """
    if data_str[0:2] == "0x":
        return data_str[2:]
    else:
        return data_str


def is_hex_digits(data_str):
    """
    data_str文字列が16進数文字列かを判定する。
    :param data_str: string
    :return:
        bool
            True if successfull, False otherwise.
    """
    hex_digits = set("0123456789abcdef")
    for char in data_str:
        if not (char in hex_digits):
            return False
    return True


def is_binary_digits(data_str):
    """
     data_str文字列が2進数文字列かを判定する。
     :param data_str: string
     :return:
         bool
             True if successfull, False otherwise.
     """
    binary_digits = set("01")
    for char in data_str:
        if not (char in binary_digits):
            return False
    return True


def make_bitmask(bits):
    """
    bits:ビット数のビットマスクを生成する。
    :param bits: int
    :return: int
    """
    seed = 0
    for i in range(bits):
        seed |= (1 << i)
    return seed


def make_bitfield_mask(start_bit, end_bit):
    """
    start_bit開始位置,end_bit終了位置でのbitfield maskを生成する。
    :param start_bit: int
    :param end_bit: int
    :return: int
    """
    if start_bit <= 0:
        return make_bitmask(end_bit+1)
    else:
        return make_bitmask(end_bit+1) & (~make_bitmask(start_bit))


def bit_flag(bit):
    return 1 << bit


def bit_flag_on(value, bit):
    return (value & bit_flag(bit)) != 0


def binary_notation(value, limit_bits):
    note = format(value, 'b')
    if len(note) < limit_bits:
        return format(value, 'b').zfill(limit_bits)
    else:
        return note[len(note)-limit_bits:]


def pure_string(data_str):
    return data_str.lower().replace(' ', '').replace('_', '').replace('-', '').translate(str.maketrans('*?', '00'))


def get_binary_array(hex_string):
    """
    １６進数文字列hex_stringを与えて、ビット列配列を返す。
    :param hex_string: １６進数文字列
    :return: ビット列配列
    """
    binary_array = []
    if not is_hex_digits(hex_string):
        return binary_array
    for i in range(len(hex_string)):
        half_byte = int(hex_string[i], 16)
        for j in range(3, -1, -1):
            if bit_flag_on(half_byte, j):
                bit_value = 1
            else:
                bit_value = 0
            binary_array.append(bit_value)
    return binary_array


def get_hexadecimal_string(binary_string):
    """
    ２進数文字列binary_stringを１６進数文字列で返す。
    :param binary_string: ２進数文字列
    :return: １６進数文字列
    """
    if not is_binary_digits(binary_string):
        return ""
    length = len(binary_string)
    start_index = 0
    hex_string=""
    while start_index < length:
        end_index = start_index + 4
        if end_index > length:
            end_index = length
        hex_string += format(int(binary_string[start_index:end_index], 2), "x")
        start_index += 4
    return hex_string


def make_binary_string_by_flag(number_of_bits, flag=True):
    """
    ビット数number_of_bitsとビットフラグflagを与えて、２進数文字列を返す。
    flag=True →　1が連続するビット数分の２進数文字列を返す。
    flag=False →　0が連続するビット数分の２進数文字列を返す。
    :param number_of_bits: ビット数
    :param flag: ビットフラグ
    :return: ２進数文字列
    """
    if flag:
        bit_value = 1
    else:
        bit_value = 0
    bit_string = ""
    for i in range(number_of_bits):
        bit_string += str(bit_value)
    return bit_string


def make_hexadecimal_zero_string(number_of_bits):
    """
    ビット数number_of_bitsを与えて対応する０ビット数の１６進数文字列を返す。
    :param number_of_bits: ビット数
    :return: ゼロ１６進数文字列
    """
    return get_hexadecimal_string(make_binary_string_by_flag(number_of_bits, False))


def zero_pad_hexadecimal_string(bits, value):
    """
    ビット数number_of_bits幅でvalue値の１６進数文字列を返す。
    :param bits: ビット幅
    :param value: 値
    :return: １６進数文字列
    """
    suffix = format(value, "b")
    length = len(suffix)
    if length > bits:
        binary_string = suffix[length - bits:]
    else:
        binary_string = make_binary_string_by_flag(bits - length, False) + suffix
    return get_hexadecimal_string(binary_string)


def uplace_degrees(val):
    degrees = math.floor(val / (60*60*10))
    minutes = math.floor((val - degrees * 60 * 60 * 10) / (60*10))
    seconds = (val - degrees * 60 * 60 * 10 - minutes * 60 * 10) / 10
    return str(degrees) + " degrees, " + str(minutes) + " minutes, " + str(seconds) + " seconds "


def display_uplace_degrees(sign_value, degrees_value, explain_str, kind_str):
    print(" " + kind_str + " sign:  " + str(sign_value) + " : " + explain_str + " " + kind_str)
    print(" " + kind_str + " value: " + str(degrees_value) + " ( 0.1 seconds ) = " + uplace_degrees(degrees_value))


def ordinal_number(num):
    remain = num % 10
    if remain == 1:
        return str(num) + "st"
    elif remain == 2:
        return str(num) + "nd"
    elif remain == 3:
        return str(num) + "rd"
    else:
        return str(num) + "th"


def floor_name(floor_value):
    """
    :param floor_value: 階層(8bits)
    :return: 階層表現
    """
    if floor_value <= 48:
        return "Underground " + str(floor_value - 50) + " floor"
    elif floor_value == 49:
        return "Underground first floor"
    elif floor_value == 50:
        return "Ground floor"
    elif (floor_value >= 51) and (floor_value <= 250):
        return ordinal_number(floor_value - 50) + " floor"


def get_floor(floor_value, floor_flag):
    """
    :param floor_value: 階層(8bits)
    :param floor_flag:中間層(1bit)
    :return:階層表現
    """
    if floor_value == 0b11111110 and floor_flag == 1:
        return "ocean floor or lake bottom or river bottom"
    elif floor_value == 0b11111111:
        if floor_flag == 1:
            return "outdoors"
        else:
            return "rooftop"
    if floor_flag == 1 and floor_value <= 249:
        return "between the " + floor_name(floor_value) + " and the " + floor_name(floor_value + 1)
    elif floor_value <= 250:
        return floor_name(floor_value)
    else:
        return ""


def get_encoding_file(path_file_name):
    """
    ファイルを与えて文字エンコード名を取得する。
    :param path_file_name: パスファイル名
    :return: ファイル・エンコーディング名
    """
    detector = UniversalDetector()
    with open(path_file_name, mode='rb') as f:
        for binary in f:
            detector.feed(binary)
            if detector.done:
                break
    detector.close()
    result = detector.result['encoding']
    if result is None:
        return ''
    else:
        return result


# ---------------------------------------------------------------------------------------------------------------------
# ucodeダンプのためのクラス定義
# ---------------------------------------------------------------------------------------------------------------------
class Ucode(object):
    # CONSTANT #
    COPY_RIGHTS = "CopyRight(C)2014,T-Engine,Ubiquitous ID Center(TRON Forum,2015-).All rights reserved."

    def __init__(self, data_str):
        self.setup_ucode(data_str)

    def setup_ucode(self, data_str):
        self.data_source = data_str
        self.ubiquitous_code_binary = []
        self.ubiquitous_code_bit = []
        if self.check_ucode_string(data_str):
            self.ubiquitous_code_str = self.ucode_string(data_str)
            self.set_array()
            self.set_bit_array()
        else:
            self.ubiquitous_code_str = ""

    def to_string(self):
        return self.ubiquitous_code_str

    def get_source(self):
        return self.data_source

    def is_ucode(self):
        if self.ubiquitous_code_str == "":
            return False
        else:
            return True

    def set_array(self):
        """precondition:
            exute before initialization of __init___
        :return:
        """
        for ch in self.ubiquitous_code_str:
            self.ubiquitous_code_binary.append(int(ch, 16))

    def set_bit_array(self):
        """
        precondition:
            excute after set_array
        :return:
        """
        for i in range(len(self.ubiquitous_code_binary)):
                for j in range(3, -1, -1):
                    if bit_flag_on(self.ubiquitous_code_binary[i], j):
                        bit_value = 1
                    else:
                        bit_value = 0
                    self.ubiquitous_code_bit.append(bit_value)

    def get_characters(self):
        return len(self.ubiquitous_code_str)

    def get_width(self):
        """
        :return: bit数を返す。
        """
        return self.get_characters() * 4

    def ucode_string(self, data_str):
        return del_hex_prefix(pure_string(data_str))

    def check_ucode_string(self, data_str):
        """
        :param data_str: ucode16進数文字列
        data_strがucode16進数文字列であるかを返す。
        :return:
            True if successfull,False otherwise
        """
        uc_str = self.ucode_string(data_str)
        '先頭文字「#」「//」で始まるものは注釈として無視'
        if uc_str[0:1] == "#":
            return False
        if uc_str[0:2] == "//":
            return False
        if len(uc_str) == 0:
            return False
        if (len(uc_str) * 4) % 128 != 0:
            return False
        if is_hex_digits(uc_str):
            return True
        else:
            return False

    def get_binary_string(self, start_bit, bit_size, separate_bit=0, separator=""):
        """
        :param start_bit: ucodeを取り出す開始ビット位置(0-128)
        :param bit_size: ucodeを取り出すビット幅
        :param separate_bit; 切り出しビット数単位
        :param separator: 切り出した位置に挿入する文字列
        :return: 取り出された２進数文字列（セパレータで編集）
        """
        binary_string = ""
        count = 0
        for i in range(start_bit, start_bit + bit_size):
            binary_string += format(self.ubiquitous_code_bit[i], '0')
            count += 1
            if count == separate_bit and separate_bit > 0:
                if i < start_bit + bit_size - 1:
                    binary_string += separator
                count = 0
        return binary_string

    def get_bit_value(self, start_bit, bit_size):
        """
        :param start_bit: ucodeを取り出す開始ビット位置(0-128)
        :param bit_size: ucodeを取り出すビット幅
        :return: 取り出された値
        """
        return int(self.get_binary_string(start_bit, bit_size), 2)

    def get_hex_string(self, start_bit, bit_size):
        """
        ucodeのstart_bit位置からbit_size幅のビットデータを16進数文字列で返す。
        :param start_bit:開始ビット位置
        :param bit_size:ビット幅
        :return:１６進数文字列
        """
        return get_hexadecimal_string(self.get_binary_string(start_bit, bit_size))

    def get_ic_binary(self, start_bit, bit_size, separate_bit=0, separator=""):
        return self.get_binary_string(self.get_ic_start()[0] + start_bit, bit_size, separate_bit, separator)

    def get_ic_bits(self, start_bit, bit_size):
        return self.get_bit_value(self.get_ic_start()[0] + start_bit, bit_size)

    def get_ic_hex(self, start_bit, bit_size):
        return self.get_hex_string(self.get_ic_start()[0] + start_bit, bit_size)

    def get_ver(self):
        """
        :return: ucodeのバージョンを返す。
        """
        return self.get_bit_value(VER_START, VER_WIDTH)

    def get_TLDc(self):
        """
        :return:  TLDc(Top Level Domain Code)を返す。
        """
        return self.get_bit_value(TLDC_START, TLDC_WIDTH)

    def get_cc(self):
        """
        :return: Class Code(cc)を返す。
        """
        return self.get_bit_value(CC_START, CC_WIDTH)

    def is_ucode(self):
        return self.ubiquitous_code_str != ""

    def is_ucode_128(self):
        """
        :return: 1if length of ucode is 128bits,return True, False otherwise
        """
        # return bit_flag_on(self.get_cc(), 3)
        return self.ubiquitous_code_bit[CC_START] == 1

    def preget_SLDc_width(self, cc):
        """
        :param cc: Class Code
        :return: タプル(SLDcのビット長, 真偽値）を返す。
        """
        if cc == CC_CLASS_A:
            return 8, True
        elif cc == CC_CLASS_B:
            return 24, True
        elif cc == CC_CLASS_C:
            return 40, True
        elif cc == CC_CLASS_D:
            return 56, True
        elif cc == CC_CLASS_E:
            return 72, True
        elif cc == CC_CLASS_F:
            return 88, True
        else:
            return 0, False

    def SLDc_width(self, cc):
        return self.preget_SLDc_width(cc)[0]

    def preget_SLDc_ic_width(self, cc):
        """
        SLDcとicの幅（ビット長）を取得する。
        :param cc:  Class Code
        :return:  タプル(SLDcビット長,icビット長,真偽値)を返す。
        """
        result = self.preget_SLDc_width(cc)
        return result[0], SLDC_PLUS_IC_WIDTH - result[0], result[1]

    def preget_ic_width(self, cc):
        """
             icの幅（ビット長）を取得する。
             :param cc:  Class Code
             :return:  タプル(icビット長,真偽値)を返す。
             """
        result = self.preget_SLDc_ic_width(cc)
        return result[1], result[2]

    def preget_ic_start(self, cc):
        """
        icの開始位置（ビット位置）を取得する。
        :param cc:  Class Code
        :return: タプル（icの開始位置, 真偽値）を返す。
        """
        result = self.preget_SLDc_width(cc)
        return SLDC_START + result[0], result[1]

    def get_SLDc_ic_width(self):
        return self.preget_SLDc_ic_width(self.get_cc())

    def get_SLDc_width(self):
        return self.preget_SLDc_width(self.get_cc())

    def get_ic_width(self):
        return self.preget_ic_width(self.get_cc())

    def get_ic_start(self):
        return self.preget_ic_start(self.get_cc())

    def ic_start(self):
        return self.get_ic_start()[1]

    def get_class_name(self):
        """
        Class名を取得する。
        :return: Class名
        """
        cc = self.get_cc()
        if cc == CC_CLASS_A:
            return "Class_A"
        elif cc == CC_CLASS_B:
            return "Class_B"
        elif cc == CC_CLASS_C:
            return "Class_C"
        elif cc == CC_CLASS_D:
            return "Class_D"
        elif cc == CC_CLASS_E:
            return "Class_E"
        elif cc == CC_CLASS_F:
            return "Class_F"
        elif cc == CC_CLASS_RESERVED_1000 or cc == CC_CLASS_RESERVED_1111:
            return "Reserved"
        else:
            return "Reserved for extension"

    def get_TLDc_name(self):
        """
        特殊用途のTLDc名を取得する。
        :return: TLDc名
        """
        tldc = self.get_TLDc()
        if tldc == TLDC_META_CODE_SPACE:
            return "Reserved ucode space ( metacode space )"
        elif tldc == TLDC_RESERVED_SPACE:
            return "Reserved space"
        elif tldc == TLDC_COMMON_VOCABULARY:
            return "Common vocabulary ucode space"
        elif tldc == TLDC_ETRON_ID:
            return "eTRON ID"
        else:
            return ""

    def get_SLDc_hex_string(self):
        """
        SLDcを１６進数文字列で返す。
        :return: SLDc１６進数文字列
        """
        sldc = self.get_SLDc_width()
        if sldc[1]:
            return self.get_hex_string(SLDC_START, sldc[0])
        else:
            return ""

    def get_SLDc_binary_string(self):
        """
        SLDcを２進数文字列で返す。
        :return: SLDc２進数文字列
        """
        sldc = self.get_SLDc_width()
        if sldc[1]:
            return self.get_binary_string(SLDC_START, sldc[0], separate_bit=4, separator="-")
        else:
            return ""

    def get_SLDc_hex_string_with_offset_width(self, offset_bit, width_bits):
        """
        SLDcの開始offset_bitオフセット位置からwidth_bitsの幅で値を１６進数文字列で取り出す。
        :param offset_bit: 開始ビットオフセット位置(SLDcのスタートビッチ位置を原点0とした相対位置）
        :param width_bits: 取り出したいSLDc範囲内のビット長
        :return: SODcの部分ビット１６進数文字列
        """
        sldc = self.get_SLDc_width()
        if not sldc[1]:
            return ""
        if offset_bit < 0 or offset_bit >= sldc[0]:
            return ""
        if offset_bit + width_bits > sldc[0]:
            limit_bits = sldc[0] - offset_bit
        else:
            limit_bits = width_bits
        return self.get_hex_string(SLDC_START+offset_bit, limit_bits)

    def get_SLDc_binary_string_with_offset_width(self, offset_bit, width_bits):
        """
             SLDcの開始offset_bitオフセット位置からwidth_bitsの幅で値を２進数文字列で取り出す。
             :param offset_bit: 開始ビットオフセット位置(SLDcのスタートビッチ位置を原点0とした相対位置）
             :param width_bits: 取り出したいSLDc範囲内のビット長
             :return: SODcの部分ビット２進数文字列
             """
        sldc = self.get_SLDc_width()
        if not sldc[1]:
            return ""
        if offset_bit < 0 or offset_bit >= sldc[0]:
            return ""
        if offset_bit + width_bits > sldc[0]:
            limit_bits = sldc[0] - offset_bit
        else:
            limit_bits = width_bits
        return self.get_binary_string(SLDC_START + offset_bit, limit_bits, separate_bit=4, separator="-")

    def get_ic_hex_string(self):
        """
        icを１６進数文字列で返す。
        :return: ic１６進数文字列
        """
        sldc_ic = self.get_SLDc_ic_width()
        if sldc_ic[2]:
            return self.get_hex_string(SLDC_START + sldc_ic[0], sldc_ic[1])
        else:
            return ""

    def get_ic_binary_string(self):
        """
        icを２進数文字列で返す。
        :return: ic２進数文字列
        """
        sldc_ic = self.get_SLDc_ic_width()
        if sldc_ic[2]:
            return self.get_binary_string(SLDC_START + sldc_ic[0], sldc_ic[1], 4, "-")
        else:
            return ""

    def get_SLDc_ic_hex_string(self):
        """
        SLDc+icの１６進数文字列を返す。
        :return: SLDc+ic１６進数文字列
        """
        return self.get_hex_string(SLDC_START, SLDC_PLUS_IC_WIDTH)

    def get_SLDc_ic_binary_string(self):
        """
        SLDc+ic２進数文字列を返す。
        :return: SLDc+ic２進数文字列
        """
        return self.get_binary_string(SLDC_START, SLDC_PLUS_IC_WIDTH, 4, "-")

    def check_special_code(self, bit_value):
        """
        ucode:0x?0-0000-00-0000-0000-0000-00000-0000-0000
        ucode:0x?F-FFFF-FF-FFFF-FFFF-FFFF-FFFF-FFFF-FFFF
        の場合、bit_value 0,1でそれぞれを判定する。
        :param bit_value: ビット値(1,0)
        :return: 結果　真偽値
        """
        length = len(self.ubiquitous_code_bit)
        if length > 128:
            length = 128
        elif length < 4:
            return False
        if bit_value == 1 or bit_value == 0:
            pass
        else:
            return False
        for i in range(4, length):
            if self.ubiquitous_code_bit[i] != bit_value:
                return False
        return True

    def is_special_code(self):
        """
        特殊コードの場合、真を返す。
        特殊コードは以下の場合（?は認知値）
        ucode:0x?0-0000-00-0000-0000-0000-00000-0000-0000
        ucode:0x?F-FFFF-FF-FFFF-FFFF-FFFF-FFFF-FFFF-FFFF
        :return: 真偽値, 種別
        """
        if self.check_special_code(0):
            return True, "reserved"
        elif self.check_special_code(1):
            return True, "reserved"
        else:
            return False, ""

    def equal(self, start_bit, bit_size, bit_array):
        """
        ucodeのstart_bit開始ビット位置からビット幅_bit_sizeがbit_arrayビット配列と一致するかを返す。
        :param start_bit: ucodeの開始ビット位置
        :param bit_size: ucodeのビット幅
        :param bit_array: 比較するビット列の配列
        :return: 一致真偽値
        """
        if len(bit_array) != bit_size:
            return False
        for i in range(bit_size):
            if self.ubiquitous_code_bit[start_bit+i] != bit_array[i]:
                return False
        return True

    def equal_SLDc(self, hex_string):
        """
        １６進数文字列hex_stringがSLDc(Domain Code)に等しいかを返す。
        :param hex_string: １６進数文字列
        :return: SLDc(Domain Code)に一致するか真偽値を返す。
        """
        bit_array = get_binary_array(hex_string)
        sldc = self.get_SLDc_width()
        if sldc[1] and len(hex_string) * 4 == sldc[0]:
            return self.equal(SLDC_START, sldc[0], bit_array)
        else:
            return False

    def equal_SLDc_header(self, hex_string):
        """
         １６進数文字列hex_stringがSLDc(Domain Code)の先頭部分に等しいかを返す。
         :param hex_string: １６進数文字列
         :return: SLDc(Domain Code)の先頭部分に一致するか真偽値を返す。
         """
        bit_array = get_binary_array(hex_string)
        sldc = self.get_SLDc_width()
        if sldc[1] and len(hex_string) * 4 <= sldc[0]:
            return self.equal(SLDC_START, len(hex_string) * 4, bit_array)
        else:
            return False

    def equal_SLDc_zero(self, cc):
        """
        クラスコードccを与えてSLDcが０値であるかを返す。
        :param cc:Class Code
        :return:真偽値
        """
        return self.equal_SLDc(make_hexadecimal_zero_string(self.SLDc_width(cc)))

    def is_authorized_standard_meta_code(self):
        """
        標準認定コードであるかを返す。
        :return:真偽値
        """
        return self.get_ver() == 0x0 and self.get_TLDc() == TLDC_META_CODE_SPACE

    def is_mu_ID(self):
        return self.is_authorized_standard_meta_code() and self.get_cc() == CC_CLASS_B

    def is_ISO_IEC_15963(self):
        return self.is_authorized_standard_meta_code() and self.get_cc() == CC_CLASS_C \
            and self.equal_SLDc_zero(CC_CLASS_C)

    def is_ISO_IEC_11784(self):
        return self.is_authorized_standard_meta_code() and self.get_cc() == CC_CLASS_C \
            and self.equal_SLDc(zero_pad_hexadecimal_string(bits=self.SLDc_width(CC_CLASS_C), value=1))

    def is_LAS300(self):
        return self.is_authorized_standard_meta_code() and self.get_cc() == CC_CLASS_D \
            and self.equal_SLDc_zero(CC_CLASS_D)

    def is_color_code(self):
        return self.is_authorized_standard_meta_code() and self.get_cc() == CC_CLASS_E \
            and self.equal_SLDc_zero(CC_CLASS_E)

    def is_RFtag03TXC(self):
        return self.is_authorized_standard_meta_code() and self.get_cc() == CC_CLASS_F \
               and self.equal_SLDc_header(make_hexadecimal_zero_string(80))

    def is_FP_code(self):
        return self.is_authorized_standard_meta_code() and self.get_cc() == CC_CLASS_D \
            and self.equal_SLDc(zero_pad_hexadecimal_string(bits=self.SLDc_width(CC_CLASS_D), value=1))

    def get_Allocation_class_of_Permanent_UID(self):
        return self.get_bit_value(self.preget_ic_start(CC_CLASS_C)[0], 8)

    def is_ISO_IEC_7816_6_class(self):
        return self.is_ISO_IEC_15963() and self.get_Allocation_class_of_Permanent_UID() == 0xe0

    def get_UID_issuer_registration_number_of_Permanent_UID(self):
        return self.get_bit_value(self.preget_ic_start(CC_CLASS_C)[0] + 8, 8)

    def get_serial_number_of_Permanent_UID(self):
        return self.get_bit_value(self.preget_ic_start(CC_CLASS_C)[0] + 16, 48)

    def get_remain_of_Parmanent_UID_without_Allocation_class(self):
        return self.get_bit_value(self.preget_ic_start(CC_CLASS_C)[0] + 8, 56)

    def get_ucode_ISO_IEC_15963(self):
        if self.is_ISO_IEC_15963():
            return self.get_hex_string(self.get_ic_start(), self.get_ic_width()[0])
        else:
            return ""

    def get_ucode_ISO_IEC_11784(self):
        if self.is_ISO_IEC_11784():
            return self.get_hex_string(self.get_ic_start(), self.get_ic_width()[0])
        else:
            return ""

    def is_uPlace(self):
        """
        国土地理院　場所情報コード:uPlaceであるかを返す。
        :return: 真偽値
        """
        return self.get_ver() == 0x0 and self.get_TLDc() == 0x1 and self.get_cc() == CC_CLASS_C \
            and self.equal_SLDc("0000000003")

    def dump_bits(self, sep=True):
        """
        分離符号有無指定で編集されたucodeを２進数文字列を返す。
        :param sep: セパレター有無（省略時は有）
        :return:編集２進数文字列
        """
        dsp_str = ""
        count = 0
        for ch in self.ubiquitous_code_bit:
            if count % 4 == 0 and count > 0 and sep:
                dsp_str += "-"
            dsp_str += format(ch, "0")
            count += 1
        return dsp_str

    def check_member_domain(self, member_domain_pattern_string):
        """
        会員向けucodeドメインのパターンを与えて一致するか（真偽値）を返す。
        :param member_domain_pattern_string: ucode１６進数文字列（-は無視、*は任意を示す）
        :return: 一致（真偽値）
        """
        success, pat, flag = exchange_pattern(member_domain_pattern_string)
        if not success:
            return False
        length = len(self.ubiquitous_code_binary)
        for i in range(0, length):
            if flag[i]:
                if self.ubiquitous_code_binary[i] != pat[i]:
                    return False
        return True

    # =============================================================================================================== #
    # uccodeダンプ メソッド
    # =============================================================================================================== #
    def dump(self):
        if not self.is_ucode():
            print("description: " + self.get_source())
            print("This code is not the ucode(i.e.,ubiquitous code).")
            return
        print("* description of ucode: " + self.get_source())
        self.predump()
        print("")

    def stamp_copyrights(self):
        print("")
        print("Specification:" + self.COPY_RIGHTS + "\n")

    def predump(self):
        # ---------------------- #
        # a sub function of dump #
        # ---------------------- #
        def get_order_string(limit, separator="-"):
            """
            limit上限の順序数を分離符号separatorで編集した文字列表現を返す部分関数
            :param limit: 順序数上限（０以上）
            :param separator: 分離符号（文字列１文字推奨）
            :return: 順序表現文字列
            """
            result = ""
            for i in range(limit):
                length = len(str(i))
                if length == 1:
                    result += "  #" + str(i)
                elif length == 2:
                    result += " #" + str(i)
                elif length == 3:
                    result += "#" + str(i)
                else:
                    result += str(i)
                if (i + 1) < limit:
                    result += separator
            return result
        # ---------------------- #
        # a sub function of dump #
        # ---------------------- #
        def get_halfbyte_string(halfbyte_string, separator="-"):
            """
            対象文字列halfbyte_stringを分離符号で編集した文字列を返す部分関数
            get_order_stringの表示間隔に合わせること
            :param halfbyte_string: 対象文字列（ucodeの１６進数文字列）
            :param separator: 分離符号
            :return: 編集文字列
            """
            result = ""
            length = len(halfbyte_string)
            for i in range(length):
                result += "   " + halfbyte_string[i]
                if (i + 1) < length:
                    result += separator
            return result

        print("Ubiquitous Code:ucode")
        # -------- #
        # 全体表示 #
        # -------- #
        print("ucode=0x" + self.to_string())
        print("      order:" + get_order_string(len(self.to_string())))
        print("hexadecimal:" + get_halfbyte_string(self.to_string()))
        print("     binary:" + self.dump_bits())
        # ------------ #
        # コード長表示 #
        # ------------ #
        msg = ""
        if self.is_ucode_128():
            if self.get_width() == 128:
                msg = "OK"
            else:
                msg = "The code width of this ucode isn't 128bits width."
        print(" code width=" + str(self.get_width()) + "bits " + msg)
        # ---------- #
        # 特殊コード #
        # ---------- #
        special_code = self.is_special_code()
        if special_code[0]:
            print("Special Code:type " + special_code[1])
        # ------------ #
        # 構成要素表示 #
        # ------------ #
        print("*version=" + hex(self.get_ver()) + \
              " : (" + self.get_binary_string(VER_START, VER_WIDTH) + ")2")
        print("*Top Level Domain Code:TLDc=" + hex(self.get_TLDc()) + " ; (" + \
              self.get_binary_string(TLDC_START, TLDC_WIDTH, 4, "-") + ")2 : " + self.get_TLDc_name())
        print("*Class Code:cc=" + hex(self.get_cc()) + " : (" + \
              self.get_binary_string(CC_START, CC_WIDTH) + ")2 : " + self.get_class_name())
        sldc_ic = self.get_SLDc_ic_width()
        if sldc_ic[2]:
            print("*Second Level Domain Code : SLDc=0x" + self.get_SLDc_hex_string() + \
                  " : width=" + str(sldc_ic[0]) + "bits")
            print("      order:" + get_order_string(len(self.get_SLDc_hex_string())))
            print("hexadecimal:" + get_halfbyte_string(self.get_SLDc_hex_string()))
            print("     binary:" + self.get_SLDc_binary_string())
            print("*Identification Code      :   ic=0x" + self.get_ic_hex_string() + \
                  " : width=" + str(sldc_ic[1]) + "bits")
            print("      order:" + get_order_string(len(self.get_ic_hex_string())))
            print("hexadecimal:" + get_halfbyte_string(self.get_ic_hex_string()))
            print("     binary:" + self.get_ic_binary_string())
        else:
            print("*SLDc + ic=0x" + self.get_SLDc_ic_hex_string() + " : width=" + str(SLDC_PLUS_IC_WIDTH) + "bits")
            print("      order:" + get_order_string(SLDC_PLUS_IC_WIDTH))
            print("hexadecimal:" + get_halfbyte_string(self.get_SLDc_ic_hex_string()))
            print("     binary:" + self.get_SLDc_ic_binary_string())
        # ---------------------------------------------------------------------------------------------------------- #
        # 特定コードの内容表示
        # ---------------------------------------------------------------------------------------------------------- #
        # ---------- #
        # μIDコード #
        # ---------- #
        if self.is_mu_ID():
            print("** Domain Name: mu ID")
            print("   ucode-mu ID (104bits)=SLDc(24bits:0x000000-0xffffff)+ic(80bits)")
            hex_mu_id = self.get_SLDc_ic_hex_string()
            print("      order:" + get_order_string(len(hex_mu_id)))
            print("hexadecimal:" + get_halfbyte_string(hex_mu_id))
            print("     binary:" + self.get_SLDc_ic_binary_string())
        # ---------------------------------------------------------------- #
        # ISO/IEC 15963 の ISO/IEC 7816-6 クラスで規定される Permanent UID #
        # ---------------------------------------------------------------- #
        if self.is_ISO_IEC_15963():
            print("** Domain Name: ISO/IEC 15963 (ISO/IEC 7816-6 class) ")
            if self.is_ISO_IEC_7816_6_class():
                print("* ISO/IEC 7816-6 class")
                print("  Allocation class:0x" + format(self.get_Allocation_class_of_Permanent_UID(), '02x'))
                print("  UID issuer registration number:0x" + \
                      format(self.get_UID_issuer_registration_number_of_Permanent_UID(), '02x'))
                print("  Serial Number:0x" + format(self.get_serial_number_of_Permanent_UID(), '012x'))
            else:
                print("  Allocation class:0x" + format(self.get_Allocation_class_of_Permanent_UID(), '02x'))
                print("  UID issuer registration number + Serial Number:0x" + \
                      format(self.get_remain_of_Parmanent_UID_without_Allocation_class(), '014x'))
        # ------------------------------ #
        # ISO/IEC 11784 RFID for animals #
        # ------------------------------ #
        if self.is_ISO_IEC_11784():
            print("** Domain Name: ISO/IEC 11784")
            print("*  Radio-frequency identification of animals")
            value = self.get_ic_bits(ANIMAL_RFID_ANIMAL_START, ANIMAL_RFID_ANIMAL_WIDTH)
            if value == 1:
                msg = "(animal)"
            else:
                msg = "(non-animal)"
            print("  Animal non-animal application(" + display_bit_with_unit(ANIMAL_RFID_ANIMAL_WIDTH) + \
                  "):" + str(value) + " " + msg)
            print("  ISO reserved Field(" + display_bit_with_unit(ANIMAL_RFID_RESERVED_WIDTH) + "):0x" + \
                  self.get_ic_hex(ANIMAL_RFID_RESERVED_START, ANIMAL_RFID_RESERVED_WIDTH))
            value = self.get_ic_bits(ANIMAL_RFID_DATA_BLOCK_START,ANIMAL_RFID_DATA_BLOCK_WIDTH)
            if value == 1:
                msg = "(data block)"
            else:
                msg = "(no data block)"
            print("  Data block flag(" + display_bit_with_unit(ANIMAL_RFID_DATA_BLOCK_WIDTH) + "):" + \
                  str(value) + " " + msg)
            print("  ISO 3166 numeric-3 country(" + display_bit_with_unit(ANIMAL_RFID_COUNTRY_WIDTH) + "):0x" + \
                  self.get_ic_hex(ANIMAL_RFID_COUNTRY_START, ANIMAL_RFID_COUNTRY_WIDTH) + " : " + \
                  str(self.get_ic_bits(ANIMAL_RFID_COUNTRY_START, ANIMAL_RFID_COUNTRY_WIDTH)))
            print("  Natinal identification code(" + \
                  display_bit_with_unit(ANIMAL_RFID_IDENTIFICATION_CODE_WIDTH) + "):0x" + \
                  self.get_ic_hex(ANIMAL_RFID_IDENTIFICATION_CODE_START, ANIMAL_RFID_IDENTIFICATION_CODE_WIDTH))
        # ------ #
        # LAS300 #
        # ------ #
        if self.is_LAS300():
            print("** Domain Name: LAS300")
            print("   LAS300:Local Area Search 300, Active RFID by E&M Corporation")
            print("*  LAS300-ID=ic(" + display_bit_with_unit(self.get_ic_width()[0]) + "):ucode-LAS300=0x" + \
                  self.get_ic_hex_string())
        # --------- #
        # ColorCode #
        # --------- #
        if self.is_color_code():
            print("** Domain Name: Colorzip code")
            print("   ColorCode developed by Colorzip Japan,Inc.")
            print("*  ic(" + display_bit_with_unit(self.get_ic_width()[0]) + "):ucode-ColorCode=0x" + \
                  self.get_ic_hex_string())
        # ------ #
        # FPcode #
        # ------ #
        if self.is_FP_code():
            print("** Domain Name: FPcode")
            print("   FP(Fine Picture)code developed by Fujitsu Limited")
            print("*  ic(" + display_bit_with_unit(self.get_ic_width()[0]) + "):ucode-FPcode=0x" + \
                  self.get_ic_hex_string())
        # --------------- #
        # RFtag03TXC code #
        # --------------- #
        if self.is_RFtag03TXC():
            print("** Domain Name: RFtag03TXC code")
            print("   RFtag03TXC, active tags developed by Nomura Engineering Co.,Ltd.")
            print("*  ucode-RFtag03TXC code=0x" + self.get_hex_string(SLDC_START + 80, 24))
            print("*    SLDc lower(8bits) : ucode-RFtag03TXC code upper  8 bits=0x" + \
                  self.get_SLDc_hex_string_with_offset_width(80, 8))
            print("*      ic(" + display_bit_with_unit(self.get_ic_width()[0]) + \
                  ")      : ucode-RFtag03TXC code lower 16 bits=0x" + self.get_ic_hex_string())

        # --------------------------------- c
        # 場所情報コードUcode（国土地理院） #
        # --------------------------------- #
        if self.is_uPlace():
            print("** uPlace : ucode for place identification enacted by Geospatial Information Authority of Japan")
            hex_str = self.get_ic_hex(UPLACE_CLASSIFICATION_START, UPLACE_CLASSIFICATION_WIDTH)
            bin_str = self.get_ic_binary(UPLACE_CLASSIFICATION_START, UPLACE_CLASSIFICATION_WIDTH)
            print("classification:0x" + hex_str + " (" + bin_str + ")2")
            sign = self.get_ic_bits(UPLACE_LATITUDE_START, UPLACE_LATITUDE_SIGN_WIDTH)
            if sign == 0:
                explain = "North"
            else:
                explain = "South"
            value = self.get_ic_bits(UPLACE_LATITUDE_VALUE_START, UPLACE_LATITUDE_VALUE_WIDTH)
            display_uplace_degrees(sign, value, explain, "latitude")
            sign = self.get_ic_bits(UPLACE_LONGITUDE_START, UPLACE_LONGITUDE_SIGN_WIDTH)
            if sign == 0:
                explain = "East"
            else:
                explain = "West"
            value = self.get_ic_bits(UPLACE_LONGITUDE_VALUE_START, UPLACE_LONGITUDE_VALUE_WIDTH)
            display_uplace_degrees(sign, value, explain, "longitude")
            floor = self.get_ic_bits(UPLACE_FLOOR_START, UPLACE_FLOOR_WIDTH)
            between_floor = self.get_ic_bits(UPLACE_BETWEEN_FLOOR_START, UPLACE_BETWEEN_FLOOR_WIDTH)
            print("floor:" + str(floor) + " (" + format(floor, 'b') + ")2 , between floor:" + str(between_floor) +
                  ": " + get_floor(floor, between_floor))
        # ---------------------------- #
        # 会員向け1ucodeドメイン割当て #
        # ---------------------------- #
        member_domain_list=[]
        member_domain_list.append(MEMBER_DOMAIN_YRP)
        member_domain_list.append(MEMBER_DOMAIN_UCTECH)
        member_domain_list.append(MEMBER_DOMAIN_S)
        member_domain_list.append(MEMBER_DOMAIN_K)
        member_domain_list.append(MEMBER_DOMAIN_UNIADEX)
        member_domain_list.append(MEMBER_DOMAIN_MTI)
        member_domain_list.append(MEMBER_DOMAIN_JTA)
        member_domain_list.append(MEMBER_DOMAIN_FBSEAJ)
        for member in member_domain_list:
            if self.check_member_domain(member[2]):
                print("* This ucode is assigned for members of TRON Forum.")
                print("  applicaiton group: " + member[0])
                print("   application type: " + member[1])
                print("    assigned domain: " + member[2])


def dump_direct_input_ucode(ucode_instance):
    """
    コマンドラインで直接ucode(1個）を指定した場合のダンプ
    :param ucode_str:  ucode表記（１６進数文字列 32文字）
    :return:
    """
    # uc = Ucode(ucode_str)

    if ucode_instance.is_ucode():
        ucode_instance.dump()
    else:
        print("This description is not ucode.")
        print(" description: " + ucode_instance.get_source())


def dump_read_file_ucode(path_file_name, encoding_name, ucode_instance):
    """
    コマンドラインでファイル(--src)を指定した場合のダンプ
    :param path_file_name: パス付ファイル名
    :param encoding_name: ファイルのエンコード名（省略時 utf-8)
    :return:
    """
    if not os.path.isfile(path_file_name):
        print("This src file does not exist.")
        print("src:" + str(path_file_name))
        exit()
    if encoding_name is None:
        encoding_name = 'utf-8'
    elif encoding_name == '':
        encoding_name = 'utf-8'

    src_f = open(path_file_name, 'r', encoding=encoding_name)
    # uc = Ucode('')
    for line in src_f:
        line = line.strip()
        line = line.rstrip('\n')
        line = line.rstrip('\r')
        ucode_instance.setup_ucode(line)
        if line[0:1] == '#' or line[0:2] == '//' or not ucode_instance.is_ucode():
            print(line)
        else:
            ucode_instance.dump()
    src_f.close()


def dump_command_line():
    """
    ucodeをダンプする。（直接指定とファイル指定をサポートする）

    :return:
    """
    parser = argparse.ArgumentParser(description='This utility dumps ucode,i.e.,ubiquitous code.')
    parser.add_argument('-u', '--ucode', help='description of ucode (hexadecimal string)')
    parser.add_argument('-s', '--src', help='source text file containing ucode (hexadecimal string)')
    parser.add_argument('-e', '--encoding', help='charactor encoding of source text file', type=str)
    parser.add_argument('--version', help='version of this utility', action='store_true')
    parser.add_argument('--author', help='display the author of this utility', action='store_true')
    args = parser.parse_args()

    # バージョン表示
    if args.version:
        print("ucode dump utility")
        print("    version: ver." + PROGRAM_VERSION)
        print("update time: " + PROGRAM_UPDATE_TIME)
        exit()

    # 作者表示
    if args.author:
        print(PROGRAM_AUTHOR + " created this utility.")
        exit()

    # ucodeオブジェクトの生成
    uc = Ucode('')

    # print(os.getcwd())

    display_flag = False
    # コマンドラインucode１個指定
    if args.ucode is not None:
        uc.setup_ucode(args.ucode)
        dump_direct_input_ucode(uc)
        display_flag = True

    # エンコード指定省略時、自動判定する。
    encoding_name = ""
    if args.encoding is None or args.encoding == "":
        if args.src is not None:
            encoding_name = get_encoding_file(args.src)

    # ファイル指定でucodeをダンプする。
    if args.src is not None:
        print("* dump file: " + args.src)
        print("*  encoding: " + encoding_name)
        print("")
        dump_read_file_ucode(args.src, encoding_name, uc)
        display_flag = True

    # ---------------------------- #
    # 仕様　著作権表示（これ大事） #
    # ---------------------------- #
    if display_flag:
        uc.stamp_copyrights()


if __name__ == "__main__":
    dump_command_line()
